// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_SYSTEM_WEB_NOTIFICATION_WEB_NOTIFICATION_TRAY_H_
#define ASH_SYSTEM_WEB_NOTIFICATION_WEB_NOTIFICATION_TRAY_H_

#include <memory>

#include "ash/ash_export.h"
#include "ash/login_status.h"
#include "ash/system/tray/tray_background_view.h"
#include "base/gtest_prod_util.h"
#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "ui/base/models/simple_menu_model.h"
#include "ui/gfx/animation/animation_container.h"
#include "ui/message_center/ui_delegate.h"
#include "ui/views/bubble/tray_bubble_view.h"

namespace aura {
class Window;
}

namespace message_center {
class MessageCenter;
class MessagePopupCollection;
class UiController;
}

namespace ash {
class AshPopupAlignmentDelegate;
class MessageCenterBubble;
class SystemTray;
class WebNotificationBubbleWrapper;
class WebNotificationImage;
class WebNotificationLabel;

// Status area tray for showing browser and app notifications. This hosts
// a MessageCenter class which manages the notification list. This class
// contains the Ash specific tray implementation.
//
// Note: These are not related to system notifications (i.e NotificationView
// generated by SystemTrayItem). Visibility of one notification type or other
// is controlled by StatusAreaWidget.
class ASH_EXPORT WebNotificationTray
    : public TrayBackgroundView,
      public message_center::UiDelegate,
      public base::SupportsWeakPtr<WebNotificationTray>,
      public ui::SimpleMenuModel::Delegate {
 public:
  WebNotificationTray(Shelf* shelf,
                      aura::Window* status_area_window,
                      SystemTray* system_tray);
  ~WebNotificationTray() override;

  static void DisableAnimationsForTest(bool disable);

  // Sets the height of the system tray bubble (or legacy notification bubble)
  // from the edge of the work area so that the web notification popups don't
  // overlap with the tray. Pass 0 if no bubble is shown.
  void SetTrayBubbleHeight(int height);

  // Returns the current tray bubble height or 0 if there is no bubble.
  int tray_bubble_height_for_test() const;

  // Returns true if it should block the auto hide behavior of the shelf.
  bool ShouldBlockShelfAutoHide() const;

  // Returns true if the message center bubble is visible.
  bool IsMessageCenterVisible() const;

  // Overridden from TrayBackgroundView.
  void UpdateAfterShelfAlignmentChange() override;
  void AnchorUpdated() override;
  base::string16 GetAccessibleNameForTray() override;
  void HideBubbleWithView(const views::TrayBubbleView* bubble_view) override;
  void ClickedOutsideBubble() override;
  bool PerformAction(const ui::Event& event) override;
  void CloseBubble() override;
  void ShowBubble(bool show_by_click) override;
  views::TrayBubbleView* GetBubbleView() override;

  // Overridden from views::TrayBubbleView::Delegate.
  void BubbleViewDestroyed() override;
  void OnMouseEnteredView() override;
  void OnMouseExitedView() override;
  base::string16 GetAccessibleNameForBubble() override;
  bool ShouldEnableExtraKeyboardAccessibility() override;
  void HideBubble(const views::TrayBubbleView* bubble_view) override;

  // Overridden from message_center::UiDelegate.
  void OnMessageCenterContentsChanged() override;
  bool ShowMessageCenter(bool show_by_click) override;
  void HideMessageCenter() override;
  bool ShowPopups() override;
  void HidePopups() override;
  bool ShowNotifierSettings() override;

  // Overridden from ui::SimpleMenuModel::Delegate.
  bool IsCommandIdChecked(int command_id) const override;
  bool IsCommandIdEnabled(int command_id) const override;
  void ExecuteCommand(int command_id, int event_flags) override;

  message_center::MessageCenter* message_center() const;

  message_center::UiController* message_center_ui_controller() {
    return message_center_ui_controller_.get();
  }

 private:
  friend class WebNotificationTrayTest;

  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, WebNotifications);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, WebNotificationPopupBubble);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest,
                           ManyMessageCenterNotifications);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, ManyPopupNotifications);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, PopupShownOnBothDisplays);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, PopupAndSystemTray);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, PopupAndAutoHideShelf);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, VisibleSmallIcon);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, QuietModeIcon);
  FRIEND_TEST_ALL_PREFIXES(WebNotificationTrayTest, CloseOnActivation);

  void UpdateTrayContent();

  // The actual process to show the message center. Set |show_settings| to true
  // if the message center should be initialized with the settings visible. Set
  // |show_by_click| to true if the message center is shown by mouse or gesture
  // click. Returns true if the center is successfully created.
  bool ShowMessageCenterInternal(bool show_settings, bool show_by_click);

  // Queries login status and the status area widget to determine visibility of
  // the message center.
  bool ShouldShowMessageCenter() const;

  // Returns true if it should show the quiet mode menu.
  bool ShouldShowQuietModeMenu(const ui::Event& event);

  // Shows the quiet mode menu.
  void ShowQuietModeMenu(const ui::Event& event);

  // Creates the menu model for quiet mode and returns it.
  ui::MenuModel* CreateQuietModeMenu();

  WebNotificationBubbleWrapper* message_center_bubble() const {
    return message_center_bubble_.get();
  }

  // Testing accessors.
  bool IsPopupVisible() const;
  MessageCenterBubble* GetMessageCenterBubbleForTest();

  aura::Window* status_area_window_;
  SystemTray* system_tray_;
  std::unique_ptr<message_center::UiController> message_center_ui_controller_;
  std::unique_ptr<WebNotificationBubbleWrapper> message_center_bubble_;
  std::unique_ptr<message_center::MessagePopupCollection> popup_collection_;
  std::unique_ptr<views::View> bell_icon_;
  std::unique_ptr<views::View> quiet_mode_icon_;
  std::unique_ptr<WebNotificationLabel> counter_;

  scoped_refptr<gfx::AnimationContainer> animation_container_ =
      new gfx::AnimationContainer();

  std::unordered_map<std::string, std::unique_ptr<WebNotificationImage>>
      visible_small_icons_;

  bool show_message_center_on_unlock_;

  bool should_update_tray_content_;

  // True when the shelf auto hide behavior has to be blocked. Previously
  // this was done by checking |message_center_bubble_| but actually
  // the check can be called when creating this object, so it would cause
  // flickers of the shelf from hidden to shown. See: crbug.com/181213
  bool should_block_shelf_auto_hide_;

  std::unique_ptr<AshPopupAlignmentDelegate> popup_alignment_delegate_;

  DISALLOW_COPY_AND_ASSIGN(WebNotificationTray);
};

}  // namespace ash

#endif  // ASH_SYSTEM_WEB_NOTIFICATION_WEB_NOTIFICATION_TRAY_H_
