// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef REMOTING_TEST_CYCLIC_FRAME_GENERATOR_H_
#define REMOTING_TEST_CYCLIC_FRAME_GENERATOR_H_

#include <map>
#include <memory>
#include <vector>

#include "base/callback.h"
#include "base/memory/ref_counted.h"
#include "base/time/default_tick_clock.h"
#include "remoting/protocol/input_event_timestamps.h"
#include "third_party/webrtc/modules/desktop_capture/desktop_capturer.h"

namespace remoting {
namespace test {

// CyclicFrameGenerator generates a sequence of frames that approximates
// properties of a real video stream when using a desktop applications. It
// loads a sequence of reference frames and switches between them with the
// specified frequency (every 2 seconds by default). Between reference frames it
// also generate frames with small changes which simulate a blinking cursor.
class CyclicFrameGenerator : public protocol::InputEventTimestampsSource {
 public:
  enum class ChangeType {
    // No changes.
    NO_CHANGES,

    // Whole screen changed.
    FULL,

    // Cursor state has changed.
    CURSOR,
  };

  struct ChangeInfo {
    ChangeInfo();
    ChangeInfo(ChangeType type, base::TimeTicks timestamp);

    ChangeType type = ChangeType::NO_CHANGES;
    base::TimeTicks timestamp;
  };

  typedef std::vector<ChangeInfo> ChangeInfoList;

  static scoped_refptr<CyclicFrameGenerator> Create();

  explicit CyclicFrameGenerator(
      std::vector<std::unique_ptr<webrtc::DesktopFrame>> reference_frames);

  void set_frame_cycle_period(base::TimeDelta frame_cycle_period) {
    frame_cycle_period_ = frame_cycle_period;
  }

  void set_cursor_blink_period(base::TimeDelta cursor_blink_period) {
    cursor_blink_period_ = cursor_blink_period;
  }

  void SetTickClock(base::TickClock* tick_clock);

  std::unique_ptr<webrtc::DesktopFrame> GenerateFrame(
      webrtc::SharedMemoryFactory* shared_memory_factory);

  ChangeType last_frame_type() { return last_frame_type_; }

  // Returns list of changes for the frame with the specified |timestamp|, which
  // should be an event timestamp generated by InputEventTimestampsSource
  // implemented in this class. Must be called in the same order in which frames
  // were generated.
  ChangeInfoList GetChangeList(base::TimeTicks timestamp);

  // InputEventTimestampsSource interface.
  protocol::InputEventTimestamps TakeLastEventTimestamps() override;

 private:
  ~CyclicFrameGenerator() override;
  friend class base::RefCountedThreadSafe<CyclicFrameGenerator>;

  std::vector<std::unique_ptr<webrtc::DesktopFrame>> reference_frames_;
  base::TickClock* clock_;
  webrtc::DesktopSize screen_size_;

  // By default switch between reference frames every 2 seconds.
  base::TimeDelta frame_cycle_period_ = base::TimeDelta::FromSeconds(2);

  // By default blink the cursor 4 times per seconds.
  base::TimeDelta cursor_blink_period_ = base::TimeDelta::FromMilliseconds(250);

  // Index of the reference frame used to render the last generated frame.
  int last_reference_frame_ = -1;

  // True if the cursor was rendered on the last generated frame.
  bool last_cursor_state_ = false;

  ChangeType last_frame_type_ = ChangeType::NO_CHANGES;

  base::TimeTicks started_time_;

  // frame_id of the frame passed to the last GetChangeList() call.
  int last_identifier_frame_ = -1;

  DISALLOW_COPY_AND_ASSIGN(CyclicFrameGenerator);
};

}  // namespace test
}  // namespace remoting

#endif  // REMOTING_TEST_CYCLIC_FRAME_GENERATOR_H_
